﻿# PyU3D stencil shadow example.
# Written by MysteriXYZ.

# Declare constants
SCREEN_WIDTH = 800
SCREEN_HEIGHT = 600
SPEED_LIMIT = 200

# Import the PyU3D package
import PyU3D as u3d
from PyU3D._local import *
import PyU3D.host as host
import PyU3D.custom as custom

# Psyco is strongly recommended to boost the framerate
import psyco
psyco.full()



# create a custom SunLight class, derived from the PyU3D.DirectionalLight class
class SunLight(u3d.DirectionalLight):

  """
  Directional light with custom behaviour.

  This directional light rotates around a custom axis to simulate sunlight.

  """

  def __init__(self, *args, **kwargs):

    super(SunLight, self).__init__(*args, **kwargs)

    self.rot_speed = 10.

    self.brightness = 1.
    self.brightness_incr = -self.rot_speed/180.


  def step(self):

    # define an arbitrary axis rotation to rotate the "sun"
    angle = self.rot_speed * host.getDeltaTime()
    self.transform(AxisRotationMatrix((-25., 30.), angle))

    self.brightness += self.brightness_incr * host.getDeltaTime()
    if self.brightness < 0.:
      self.brightness = 0.
      self.brightness_incr *= -1.
    elif self.brightness > 1.:
      self.brightness = 1.
      self.brightness_incr *= -1.

    self.setColor(
                  70*self.brightness + self.brightness**4 * 58,
                  35*self.brightness**2 + self.brightness**4 * 93,
                  18*self.brightness**4 + self.brightness**4 * 100
                 )

    setBackgroundColor(
                        self.brightness**3 * 120,
                        self.brightness**2 * 120,
                        self.brightness * 200
                      )

    self.update()



class Octahedron(custom.SphereModel):

  """
  Rotating octahedron with two shadow-enabled lights attached to it to light up
  the scene "at night".

  """

  def __init__(self, *args, **kwargs):

    super(Octahedron, self).__init__(resolution=4, *args, **kwargs)

    self.rot_angle = -20.

    # Variables needed to smoothly power up/down the attached lights

    self.power = 0.
    self.powering_up = False
    self.powering_down = False
    self.powerupdown_time = 2.

    # Create the light sources

    self.light1 = u3d.PointLight(r=0, g=0, b=0)
    self.light1_to_self = TranslationMatrix(0., 1.1, 0.)
    self.light1_shape = custom.SphereModel(resolution=20, scalx=2.)
    self.light1_shape.getMaterial().setDiffuseColor(200, 200, 100)
    self.light2 = u3d.PointLight(r=0, g=0, b=0)
    self.light2_to_self = TranslationMatrix(0., -1.1, 0.)
    self.light2_shape = custom.SphereModel(resolution=20, scalx=2.)
    self.light2_shape.getMaterial().setDiffuseColor(100, 100, 200)


  def updateLightIntensity(self):

    """
    Update the intensity of the attached lights while the octahedron is
    powering up/down.

    """

    self.light1.setRange(200*self.power)
    self.light1.setColor(100*self.power, 100*self.power, 50*self.power)
    self.light1_shape.getMaterial().setEmissiveColor(
                                                      200*self.power,
                                                      200*self.power,
                                                      100*self.power
                                                    )

    self.light2.setRange(200*self.power)
    self.light2.setColor(50*self.power, 50*self.power, 100*self.power)
    self.light2_shape.getMaterial().setEmissiveColor(
                                                      100*self.power,
                                                      100*self.power,
                                                      200*self.power
                                                    )


  def step(self):

    # get the time (in seconds) it currently takes to process a step of the
    # program
    delta_time = host.getDeltaTime()

    # rotate the octahedron around its Z-axis
    self.yaw += self.rot_angle * delta_time
    self.update()

    # power up or down (to control the intensity of the attached lights),
    # according to the brightness of the "daylight"
    if self.powering_up:
      self.power = min(1., self.power + delta_time/self.powerupdown_time)
      if self.power == 1.:
        self.powering_up = False
      self.updateLightIntensity()
    elif self.powering_down:
      self.power = max(0., self.power - delta_time/self.powerupdown_time)
      if self.power == 0.:
        self.powering_down = False
      self.updateLightIntensity()
    elif sunlight.brightness < .6 and sunlight.brightness_incr < 0. \
      and self.power == 0.:
      self.powering_up = True
    elif sunlight.brightness > .3 and sunlight.brightness_incr > 0. \
      and self.power == 1.:
      self.powering_down = True

    # attach the lights to the octahedron

    self.light1.alignTo(self, self.light1_to_self)
    self.light1.update()
    self.light1_shape.alignTo(self, self.light1_to_self)
    self.light1_shape.update()
    self.light2.alignTo(self, self.light2_to_self)
    self.light2.update()
    self.light2_shape.alignTo(self, self.light2_to_self)
    self.light2_shape.update()



# Set U3D options

u3d.setLog("log.txt")
u3d.setZBufferFormat(32)
u3d.setMultiSampleType(0)


# initialize U3D and its host
host.init(
          SCREEN_WIDTH, SCREEN_HEIGHT, SPEED_LIMIT,
          "PyU3D Stencil Shadow Example",
          fullscreen=False
         )


# create an object to handle keyboard input
keyb = host.Keyboard()

# create an object to handle the mouse
mouse = host.Mouse()

# hide the cursor
mouse.setVisibility(False)

# should problems arise due to the cursor being hidden, try to use a transparent
# cursor instead of hiding it; comment out the line above and de-comment the
# line below
##mouse.setCursor(None)


setAmbientLight(50, 50, 50)


# Load the textures

wall_tex = Texture("Gfx/dwall06.JPG")
concrete_tex = Texture("Gfx/conc02.jpg")
tile_tex = Texture("Gfx/tiles01.jpg")


# Create the scene objects

camera = u3d.Camera(SCREEN_WIDTH, SCREEN_HEIGHT, z=50., pitch=25., yaw=-30.)

sunlight = SunLight()

ground = u3d.Floor(
                    length=200., width=200.,
                    x=-150., y=-50.,
                    texture=tile_tex, u2=50, v2=50,
                    segs_x=10, segs_y=10
                  )


# Create a number of walls to cast and receive shadows

walls = []

for i in range(5):

  wall = custom.BoxModel(
                          length=20., width=5., height=10.,
                          x=-50.+i*15., y=100.,
                          yaw=45.,
                          texture={
                                    "back":wall_tex,
                                    "front":wall_tex,
                                    "left":wall_tex,
                                    "right":wall_tex,
                                    "top":concrete_tex,
                                    "bottom":concrete_tex
                                  }
                        )

  wall.setTexCoords("back", u2=2.)
  wall.setTexCoords("front", u2=2.)
  wall.setTexCoords("left", u2=.5)
  wall.setTexCoords("right", u2=.5)
  wall.setTexCoords("top", u2=2., v2=.5)

  walls.append(wall)


# create a rotating octahedron to light up the scene "at night"
octahedron = Octahedron(
                        x=-60., y=60., z=25.,
                        scalx=15.,
                        texture=concrete_tex
                       )

# use per-pixel lighting for the octahedron, if possible
if getSupportedPSVersion() >= 1.4:
  octahedron.getMaterial().applyPABMapping(
                                            (
                                              sunlight,
                                              octahedron.light1,
                                              octahedron.light2
                                            ),
                                            None,
                                            0.
                                          )


# Set up shadows

# First, at least one light source needs to have shadows enabled

sunlight.enableShadows()
octahedron.light1.enableShadows()
octahedron.light2.enableShadows()

# Next, model objects that should cast shadows need to have their geometry
# optimized to ensure correct looking shadows

octahedron.optimizeForShadows()
for wall in walls:
  wall.optimizeForShadows()

# Then, define which model objects should cast shadows for which light source(s)

octahedron.setShadowCasting(sunlight)
octahedron.setShadowCasting(octahedron.light1)
octahedron.setShadowCasting(octahedron.light2)
for wall in walls:
  wall.setShadowCasting(sunlight)
  wall.setShadowCasting(octahedron.light1)
  wall.setShadowCasting(octahedron.light2)

# Finally, define which objects should receive shadows for which light source(s)

ground.setShadowReceiving(sunlight)
ground.setShadowReceiving(octahedron.light1)
ground.setShadowReceiving(octahedron.light2)
octahedron.light1_shape.setShadowReceiving(octahedron.light2)
octahedron.light2_shape.setShadowReceiving(octahedron.light1)
for wall in walls:
  wall.setShadowReceiving(sunlight)
  wall.setShadowReceiving(octahedron.light1)
  wall.setShadowReceiving(octahedron.light2)


# check if the hardware supports stencil shadows
shadows_supported = getShadowSupport()


# set a font
arial12 = Font("Arial", 12, BOLDITALIC)

if not shadows_supported:
  arial12.setColor(255, 100, 0)



def main():

  # main loop
  while True:

    # process the current step of the game
    host.step()

    # allow pressing the <Escape> key to end the game
    if keyb.keyIsPressed("esc"):
      host.exit()

    # update the transformation and projection of the camera
    camera.update()

    # perform the current step of the sunlight behaviour
    sunlight.step()

    # perform the current step of the octahedron behaviour
    octahedron.step()

    # Display important statistics

    arial12.draw(10, 10, "fps: "+str(host.getFPS()))
    arial12.draw(10, 40, "triangles: "+str(getDrawnTriangleCount()))
    arial12.draw(10, 70, "draw calls: "+str(u3d.getDrawCallCount()))

    # notify the user if shadows are not supported by the hardware
    if not shadows_supported:
      arial12.draw(
                    10, 100,
                    "Sadly this PC does not support the use of shadows, so"
                    + " this example will not work as intended."
                  )


if __name__ == '__main__':
  main()
